/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/macros/block_annotation.hpp
 *
 * $Id: block_annotation.hpp,v 1.9 2005/06/23 09:54:21 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @brief [LEVEL: beta] The @ref DIAGNOSTICS_BASE_BLOCK_ENTER and
 * @ref DIAGNOSTICS_BASE_BLOCK_EXIT macros.
 *
 * @test diagnostics/macros/block_annotation.t.cpp
 */
#ifndef DIAGNOSTICS__MACROS__BLOCK_ANNOTATION_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__MACROS__BLOCK_ANNOTATION_HPP__INCLUDE_GUARD

// ::diagnostics::logging_facility::log
#include <diagnostics/frame/logging_facility.hpp>

// basefile and the like
#include <diagnostics/frame/pp_config.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;
INTERNAL_NAMESPACE_BEGIN;

/**
 * @brief helper for enter/exit log-messages. Constructor sends the enter,
 * destructor sends the exit record.
 */
template<Type_t TYPE_ENTER,Type_t TYPE_EXIT,Level_t LEVEL,int NR_WHAT,int LINE>
class Block_Guard
{
public:
    inline Block_Guard(::std::string const & str_what,
		       char const * const base_file_name,
		       char const * const file_name) :
	m_str_what(str_what),
	m_base_file_name(base_file_name),
	m_file_name(file_name)
    {
	::DIAGNOSTICS_NAMESPACE::Logging_Facility::log
	      (LEVEL, 
	       TYPE_ENTER,
	       NR_WHAT,
	       m_str_what,
	       m_base_file_name,
	       m_file_name,
	       LINE);
    }
    inline ~Block_Guard()
    {
	::DIAGNOSTICS_NAMESPACE::Logging_Facility::log
	      (LEVEL, 
	       TYPE_EXIT,
	       NR_WHAT,
	       m_str_what,
	       m_base_file_name,
	       m_file_name,
	       LINE);
    }
private:
    ::std::string const m_str_what;
    char const * const m_base_file_name;
    char const * const m_file_name;
};

INTERNAL_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

/**
 * @brief Sets up block guard. 
 *
 * @note The line/file/basefile information is taken from the entering position
 *
 * @param LEVEL the log level of type @ref diagnostics::Level_t to be
 * used int the entering/exit messages
 * @param TYPE_ENTER the type of log-message to be generated when
 * entering the block (of type @ref diagnostics::Type_t)
 * @param TYPE_EXIT the type of log-message to be generated when
 * exiting the block (of type @ref diagnostics::Type_t)
 * @param NR_WHAT the client specific nr_what field to be used in the
 * entering/exit log message
 * @param STR_WHAT the client specific str_what field to be used in
 * the entering/exit log message
 *
 */
#define DIAGNOSTICS_BASE_BLOCK_GUARD(LEVEL,TYPE_ENTER,TYPE_EXIT,NR_WHAT,STR_WHAT) \
          ::DIAGNOSTICS_NAMESPACE::INTERNAL_NAMESPACE::Block_Guard< \
                          (TYPE_ENTER), \
                          (TYPE_EXIT), \
                          (LEVEL), \
                          (NR_WHAT), \
                          __LINE__ > \
				diagnostics__internal__block_guard((STR_WHAT), (DIAGNOSTICS_BASE_FILE), __FILE__)


/**
 * @brief Sets up block enter/exit combination. 
 *
 * Equivalent to { @ref DIAGNOSTICS_BASE_BLOCK_GUARD(LEVEL,TYPE_ENTER,TYPE_EXIT,NAME,WHAT) { ... } }
 */
#define DIAGNOSTICS_BASE_BLOCK_ENTER(LEVEL,TYPE_ENTER,TYPE_EXIT,NR_WHAT,STR_WHAT) \
          do { DIAGNOSTICS_BASE_BLOCK_GUARD((LEVEL),(TYPE_ENTER),(TYPE_EXIT),(NR_WHAT),(STR_WHAT)); { 
/**
 * @brief closes a @ref DIAGNOSTICS_BASE_BLOCK_ENTER.
 */
#define DIAGNOSTICS_BASE_BLOCK_EXIT } } while(false)



#endif

// vim:ts=4:sw=4
