/*
Copyright (c) 2002-2005, Dennis M. Sosnoski.
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.
 * Neither the name of JiBX nor the names of its contributors may be used
   to endorse or promote products derived from this software without specific
   prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package org.jibx.runtime;

import java.io.PrintStream;
import java.io.PrintWriter;

/**
 * Binding exception class. This is used for all types of errors that
 * can be generated by the runtime.
 *
 * @author Dennis M. Sosnoski
 * @version 1.0
 */

public class JiBXException extends Exception
{
    /** Exception that caused this exception. */
    private Throwable m_rootCause;
    
    /**
     * Constructor from message.
     * 
     * @param msg message describing the exception condition
     */
    public JiBXException(String msg) {
        super(msg);
    }
    
    /**
     * Constructor from message and wrapped exception.
     * 
     * @param msg message describing the exception condition
     * @param root exception which caused this exception
     */
    public JiBXException(String msg, Throwable root) {
        super(msg);
        m_rootCause = root;
    }
    
    /**
     * Get root cause exception.
     *
     * @return exception that caused this exception
     */
    public Throwable getRootCause() {
        return m_rootCause;
    }

    /**
     * Print stack trace to standard error. This is an override of the base
     * class method to implement exception chaining.
     */
    public void printStackTrace() {
        printStackTrace(System.err);
    }

    /**
     * Print stack trace to stream. This is an override of the base class method
     * to implement exception chaining.
     * 
     * @param s stream for printing stack trace
     */
    public void printStackTrace(PrintStream s) {
        if (m_rootCause == null) {
            super.printStackTrace(s);
        } else {
            s.println(getMessage());
            m_rootCause.printStackTrace(s);
            s.println();
        }
    }

    /**
     * Print stack trace to writer. This is an override of the base class method
     * to implement exception chaining.
     * 
     * @param s writer for printing stack trace
     */
    public void printStackTrace(PrintWriter s) {
        if (m_rootCause == null) {
            super.printStackTrace(s);
        } else {
            s.println(getMessage());
            m_rootCause.printStackTrace(s);
            s.println();
        }
    }
}