/*
 * Copyright 2012 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "friendsdispatcher.h"
#include <glib-object.h>

/*!
 * \qmltype FriendsDispatcher
 * \inqmlmodule Friends 0.1
 * \ingroup Friends
 *
 * \brief Interface to the Friends dispatcher
 *
 * FriendsDispatcher provides an interface for applications to use the Friends
 * dispatcher
 *
 * Example of use:
 *
 * \qml
 * Item {
 *     FriendsDispatcher {
 *         id: friends
 *         onSendComplete: {
 *             if (success) {
 *                  activitySpinner.visible = false;
 *                  entry.text = "";
 *                  console.log ("Send completed successfully");
 *              } else {
 *                  activitySpinner.visible = false;
 *                  console.log ("Send failed: " + errorMessage);
 *              }
 *         }
 *
 *     }
 *
 *     TextArea {
 *          id: entry
 *          height: units.gu(4)
 *          placeholderText: i18n.tr("Compose")
 *          autoSize: true
 *          ActivityIndicator {
 *              id: activitySpinner
 *              objectName: "activitySpinner"
 *              anchors.centerIn: entry
 *              visible: false
 *              running: visible
 *          }
 *      }
 *      Button {
 *          id: sendButton
 *          width: units.gu(9)
 *          height: units.gu(4)
 *          text: i18n.tr("Send")
 *          onClicked: {
 *              console.log ("Post: " + entry.text);
 *              activitySpinner.visible = true
 *              var account = 1; //The account ID from Ubuntu.OnlineAccounts
 *              friends.sendForAccountAsync(account, entry.text);
 *          }
 *      }
 *
 * }
 * \endqml
 *
 */

FriendsDispatcherQml::FriendsDispatcherQml(QObject *parent) :
    QObject(parent),
    m_dispatcher(0)
{
    GError* error = NULL;
    m_dispatcher = friends_dispatcher_new(&error);
    if (error != NULL) 
        g_error_free (error);
}

FriendsDispatcherQml::~FriendsDispatcherQml()
{
    g_object_unref(m_dispatcher);
}

QString FriendsDispatcherQml::urlShorten(const QString& url) const
{
    QByteArray byte_array = url.toUtf8();
    GError* error = NULL;

    gchar* shortUrl = friends_dispatcher_shorten(m_dispatcher, byte_array.data(), &error);
    if (error != NULL)
        g_error_free (error);
    return shortUrl;
}

QStringList FriendsDispatcherQml::featuresForProtocol(const QString& protocol) const
{
    QStringList features;
    int length;
    QByteArray byte_array = protocol.toUtf8();
    GError* error = NULL;

    gchar** c_features = friends_dispatcher_features(m_dispatcher, byte_array.data(), &length, &error);
    for (int i = 0; i < length; i++) {
        features.append(QString::fromUtf8(c_features[i]));
    }
    if (error != NULL) 
        g_error_free (error);
    return features;
}

void FriendsDispatcherQml::uploadForAccount(const int& account_id, const QString& uri, const QString& description) const
{
    QByteArray path = uri.toUtf8();
    QByteArray desc = description.toUtf8();
    GError* error = NULL;
    friends_dispatcher_upload(m_dispatcher, account_id, path.data(), desc.data(), &error);

    if (error != NULL) 
        g_error_free (error);
}

void FriendsDispatcherQml::sendForAccount(const int& account_id, const QString& message) const
{
    QByteArray msg = message.toUtf8();
    uint acct = (uint)account_id;
    GError* error = NULL;
    friends_dispatcher_send_message(m_dispatcher, &acct, msg.data(), &error);

    if (error != NULL)
        g_error_free (error);
}

void FriendsDispatcherQml::send(const QString& message) const
{
    QByteArray msg = message.toUtf8();
    GError* error = NULL;
    friends_dispatcher_send_message(m_dispatcher, NULL, msg.data(), &error);

    if (error != NULL)
        g_error_free (error);
}

void FriendsDispatcherQml::sendReply(const int& account_id, const QString& message_id, const QString& message) const
{
    QByteArray msg = message.toUtf8();
    QByteArray msg_id = message_id.toUtf8();
    GError* error = NULL;
    friends_dispatcher_send_reply(m_dispatcher, account_id, msg_id.data(), msg.data(), &error);

    if (error != NULL)
        g_error_free (error);
}

void FriendsDispatcherQml::retweetCompleteCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
    Q_UNUSED(source_object);

    FriendsDispatcherQml* dispatcher = (FriendsDispatcherQml*) user_data;
    if (dispatcher) {
        gchar* result;
        GError* error = NULL;
        bool success = TRUE;
        friends_dispatcher_retweet_finish(dispatcher->m_dispatcher, res, &result, &error);
        QString errorMessage;
        if (result == NULL || error != NULL) {
            success = FALSE;
            errorMessage = QString::fromUtf8(error->message);
        }
        Q_EMIT dispatcher->retweetComplete(QString::fromUtf8(result), success, errorMessage);
        if (error != NULL)
            g_error_free (error);
    }
}

/*!
 * \qmlmethod object FriendsDispatcher::retweetAsync(int accountId, string messageId)
 *
 * Retweets a post for the specified \a messageId. Use retweetComplete
 * callback to check for success.
 * \list
 * \li \c accountId is the accountId, as provided by Ubuntu.OnlineAccounts
 * \li \c messageId for the post to retweet
 * \endlist
 *
 * \sa retweetComplete
 */
void FriendsDispatcherQml::retweetAsync(const int& account_id, const QString& message_id) const
{
    QByteArray msg_id = message_id.toUtf8();
    friends_dispatcher_retweet_async(m_dispatcher, account_id, msg_id.data(),
                                 &retweetCompleteCallback, (gpointer) this);
}

void FriendsDispatcherQml::retweet(const int& account_id, const QString& message_id) const
{
    QByteArray msg_id = message_id.toUtf8();
    GError* error = NULL;
    friends_dispatcher_retweet(m_dispatcher, account_id, msg_id.data(), &error);

    if (error != NULL)
        g_error_free (error);
}

void FriendsDispatcherQml::likeCompleteCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
    Q_UNUSED(source_object);

    FriendsDispatcherQml* dispatcher = (FriendsDispatcherQml*) user_data;
    if (dispatcher) {
        gchar* result;
        GError* error = NULL;
        bool success = TRUE;
        friends_dispatcher_like_finish(dispatcher->m_dispatcher, res, &result, &error);
        QString errorMessage;
        if (result == NULL || error != NULL) {
            success = FALSE;
            errorMessage = QString::fromUtf8(error->message);
        }
        Q_EMIT dispatcher->likeComplete(QString::fromUtf8(result), success, errorMessage);
        if (error != NULL)
            g_error_free (error);
    }
}

/*!
 * \qmlmethod object FriendsDispatcher::likeAsync(int accountId, string messageId)
 *
 * Likes a post for the specified \a messageId. Use likeComplete
 * callback to check for success.
 * \list
 * \li \c accountId is the accountId, as provided by Ubuntu.OnlineAccounts
 * \li \c messageId for the post to like
 * \endlist
 *
 * \sa likeComplete
 */
void FriendsDispatcherQml::likeAsync(const int& account_id, const QString& message_id) const
{
    QByteArray msg_id = message_id.toUtf8();
    friends_dispatcher_like_async(m_dispatcher, account_id, msg_id.data(),
                                 &likeCompleteCallback, (gpointer) this);
}

void FriendsDispatcherQml::like(const int& account_id, const QString& message_id) const
{
    QByteArray msg_id = message_id.toUtf8();
    GError* error = NULL;
    friends_dispatcher_like(m_dispatcher, account_id, msg_id.data(), &error);

    if (error != NULL)
        g_error_free (error);
}

void FriendsDispatcherQml::unlikeCompleteCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
    Q_UNUSED(source_object);

    FriendsDispatcherQml* dispatcher = (FriendsDispatcherQml*) user_data;
    if (dispatcher) {
        gchar* result;
        GError* error = NULL;
        bool success = TRUE;
        friends_dispatcher_unlike_finish(dispatcher->m_dispatcher, res, &result, &error);
        QString errorMessage;
        if (result == NULL || error != NULL) {
            success = FALSE;
            errorMessage = QString::fromUtf8(error->message);
        }
        Q_EMIT dispatcher->unlikeComplete(QString::fromUtf8(result), success, errorMessage);
        if (error != NULL)
            g_error_free (error);
    }
}

/*!
 * \qmlmethod object FriendsDispatcher::unlikeAsync(int accountId, string messageId)
 *
 * Un-likes a post for the specified \a messageId. Use unlikeComplete
 * callback to check for success.
 * \list
 * \li \c accountId is the accountId, as provided by Ubuntu.OnlineAccounts
 * \li \c messageId for the post to un-like
 * \endlist
 *
 * \sa unlikeComplete
 */
void FriendsDispatcherQml::unlikeAsync(const int& account_id, const QString& message_id) const
{
    QByteArray msg_id = message_id.toUtf8();
    friends_dispatcher_unlike_async(m_dispatcher, account_id, msg_id.data(),
                                 &unlikeCompleteCallback, (gpointer) this);
}

void FriendsDispatcherQml::unlike(const int& account_id, const QString& message_id) const
{
    QByteArray msg_id = message_id.toUtf8();
    GError* error = NULL;
    friends_dispatcher_unlike(m_dispatcher, account_id, msg_id.data(), &error);

    if (error != NULL)
        g_error_free (error);
}

void FriendsDispatcherQml::uploadCompleteCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
    Q_UNUSED(source_object);

    FriendsDispatcherQml* dispatcher = (FriendsDispatcherQml*) user_data;
    if (dispatcher) {
        gchar* result_url;
        GError* error = NULL;
        bool success = TRUE;
        friends_dispatcher_upload_finish(dispatcher->m_dispatcher, res, &result_url, &error);
        QString errorMessage;
        if (result_url == NULL || error != NULL) {
            success = FALSE;
            errorMessage = QString::fromUtf8(error->message);
        }
        Q_EMIT dispatcher->uploadComplete(QString::fromUtf8(result_url), success, errorMessage);
        if (error != NULL) 
            g_error_free (error);
    }
}

/*!
 * \qmlmethod object FriendsDispatcher::uploadForAccountAsync(int accountId, string uri, string description)
 *
 * Uploads file from \a uri to the specified account, use uploadComplete
 * callback to check for success.
 * \list
 * \li \c accountId is the accountId, as provided by Ubuntu.OnlineAccounts
 * \li \c uri is a URI to an image to upload, for example file:///tmp/example.jpg
 * \li \c description is a description to be included in the post
 * \endlist
 *
 * \sa uploadComplete
 */
void FriendsDispatcherQml::uploadForAccountAsync(const int& account_id, const QString& uri, const QString& description) const
{
    QByteArray path = uri.toUtf8();
    QByteArray desc = description.toUtf8();
    friends_dispatcher_upload_async(m_dispatcher, account_id, path.data(), desc.data(),
                                 &uploadCompleteCallback, (gpointer) this);
}

void FriendsDispatcherQml::sendCompleteCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
    Q_UNUSED(source_object);

    FriendsDispatcherQml* dispatcher = (FriendsDispatcherQml*) user_data;
    if (dispatcher) {
        gchar* result;
        GError* error = NULL;
        bool success = TRUE;
        friends_dispatcher_send_message_finish(dispatcher->m_dispatcher, res, &result, &error);
        QString errorMessage;
        if (result == NULL || error != NULL) {
            success = FALSE;
            errorMessage = QString::fromUtf8(error->message);
        }
        Q_EMIT dispatcher->sendComplete(QString::fromUtf8(result), success, errorMessage);
        if (error != NULL) 
            g_error_free (error);
    }
}
/*!
 * \qmlmethod object FriendsDispatcher::sendAsync(string message)
 *
 * Posts a status update to all enabled account, use sendComplete
 * callback to check for success.
 *
 * \sa sendComplete
 */
void FriendsDispatcherQml::sendAsync(const QString& message) const
{
    QByteArray msg = message.toUtf8();
    friends_dispatcher_send_message_async(m_dispatcher, NULL, msg.data(),
                                 &sendCompleteCallback, (gpointer) this);
}

/*!
 * \qmlmethod object FriendsDispatcher::sendForAccountAsync(int accountId, string message)
 *
 * Posts a status update to the specified account, use sendComplete
 * callback to check for success.
 *
 * \sa sendComplete
 */
void FriendsDispatcherQml::sendForAccountAsync(const int& account_id, const QString& message) const
{
    QByteArray msg = message.toUtf8();
    uint acct = (uint)account_id;
    friends_dispatcher_send_message_async(m_dispatcher, &acct, msg.data(),
                                 &sendCompleteCallback, (gpointer) this);
}

void FriendsDispatcherQml::sendReplyCompleteCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
    Q_UNUSED(source_object);

    FriendsDispatcherQml* dispatcher = (FriendsDispatcherQml*) user_data;
    if (dispatcher) {
        gchar* result;
        GError* error = NULL;
        bool success = TRUE;
        friends_dispatcher_send_reply_finish(dispatcher->m_dispatcher, res, &result, &error);
        QString errorMessage;
        if (result == NULL || error != NULL) {
            success = FALSE;
            errorMessage = QString::fromUtf8(error->message);
        }
        Q_EMIT dispatcher->sendComplete(QString::fromUtf8(result), success, errorMessage);
        if (error != NULL)
            g_error_free (error);
    }
}

/*!
 * \qmlmethod object FriendsDispatcher::sendReplyAsync(int accountId, string messageId, string message)
 *
 * Posts a reply status update to all enabled account, use sendComplete
 * callback to check for success.
 *
 * \sa sendComplete
 */
void FriendsDispatcherQml::sendReplyAsync(const int& account_id, const QString& message_id, const QString& message) const
{
    QByteArray msg = message.toUtf8();
    QByteArray msg_id = message_id.toUtf8();
    friends_dispatcher_send_reply_async(m_dispatcher, account_id, msg_id.data(), msg.data(),
                                 &sendReplyCompleteCallback, (gpointer) this);
}

void FriendsDispatcherQml::refresh() const
{
    GError* error = NULL;
    friends_dispatcher_refresh(m_dispatcher, &error);

    if (error != NULL)
        g_error_free (error);
}

/*!
 * \qmlsignal FriendsDispatcher::sendComplete(bool success, string errorMessage)
 *
 * Emitted when any of the async send methods completes.  Check \a success
 * for success or failure and \a errorMessage contains the error string
 * returned from the dispatcher.
 */

/*!
 * \qmlsignal FriendsDispatcher::uploadComplete(bool success, string errorMessage)
 *
 * Emitted when uploadForAccountAsync completes.  Check \a success
 * for success or failure and \a errorMessage contains the error string
 * returned from the dispatcher.
 */

/*!
 * \qmlsignal FriendsDispatcher::likeComplete(bool success, string errorMessage)
 *
 * Emitted when likeAsync completes.  Check \a success
 * for success or failure and \a errorMessage contains the error string
 * returned from the dispatcher.
 */


/*!
 * \qmlsignal FriendsDispatcher::unlikeComplete(bool success, string errorMessage)
 *
 * Emitted when unlikeAsync completes.  Check \a success
 * for success or failure and \a errorMessage contains the error string
 * returned from the dispatcher.
 */

/*!
 * \qmlsignal FriendsDispatcher::retweetComplete(bool success, string errorMessage)
 *
 * Emitted when retweetAsync completes.  Check \a success
 * for success or failure and \a errorMessage contains the error string
 * returned from the dispatcher.
 */
